/*************************************************************************
 *
 * Copyright (c) 1997-2001 Jtel Corporation or its subsidiaries.
 * All rights reserved.
 *
 * DESC:
 *
 *  Circular queue library for buffer management and general purpose
 *
 * CAUTION:
 *
 *  CqUsedSize(q) + CqUnusedSize(q) + 1 == q.Size
 *
 ************************************************************************/
#include "lib_CircularQueue.h"



/*************************************************************************
 * Function Declarations
 ************************************************************************/
void  CqInit(TCircularQueue* q, uint size);
void  CqInitWithExistingBuf(TCircularQueue* q, void* buf, uint size);;
void  CqSync(TCircularQueue* q, uint front, uint rear);
void  CqFree(TCircularQueue* q);
void  CqCopy(TCircularQueue* dst, TCircularQueue* src);
void  CqAssign(TCircularQueue* dst, TCircularQueue* src);
uint  CqUsedSize(TCircularQueue* q);
uint  CqUnusedSize(TCircularQueue* q);
bool  CqPutByte(TCircularQueue* q, char data);
bool  CqGetByte(TCircularQueue* q, char* data);
void  CqPutBuf(TCircularQueue* q, void* buf, uint bufsize, uint* realsize);
void  CqGetBuf(TCircularQueue* q, void* buf, uint bufsize, uint* realsize);
void  CqMoveBuf(TCircularQueue* q1, TCircularQueue* q2, uint size, uint* realsize);
bool  CqPutBufAll(TCircularQueue* q, void* buf, uint bufsize);
bool  CqGetBufAll(TCircularQueue* q, void* buf, uint bufsize);
bool  CqMoveBufAll(TCircularQueue* q1, TCircularQueue* q2);
char* CqParse(TCircularQueue* q, char delimitator);
char* CqParse2(TCircularQueue* q, char* delimitators, int num);



/*************************************************************************
 * Functions
 ************************************************************************/

void CqAssign(TCircularQueue* dst, TCircularQueue* src)
{
	dst->ExistingBuf = src->ExistingBuf;
	dst->Front       = src->Front;
	dst->Rear        = src->Rear;
	dst->Size        = src->Size;
	dst->Buf         = src->Buf;
}


void CqCopy(TCircularQueue* dst, TCircularQueue* src)
{
	dst->ExistingBuf = src->ExistingBuf;
	dst->Front       = src->Front;
	dst->Rear        = src->Rear;
	dst->Size        = src->Size;
	MoveBlock((byte*)src->Buf, (byte*)dst->Buf, src->Size);
}


void CqInit(TCircularQueue* q, uint size)
{
	q->ExistingBuf = FALSE;
	q->Front = 0;
	q->Rear = 0;
	q->Size = size;
	q->Buf = (char*)sLmalloc(size);
}


void CqInitWithExistingBuf(TCircularQueue* q, void* buf, uint size)
{
	q->ExistingBuf = TRUE;
	q->Front = 0;
	q->Rear = 0;
	q->Size = size;
	q->Buf = buf;
}


void CqFree(TCircularQueue* q)
{
	if ( q->ExistingBuf == FALSE && q->Buf != NULL )
		sLfree(q->Buf);
}


uint CqUsedSize(TCircularQueue* q)
{
	if ( q->Rear > q->Front ) return (q->Rear - q->Front);
	else if ( q->Rear < q->Front ) return (q->Rear + q->Size - q->Front);
	else return 0;
}


uint CqUnusedSize(TCircularQueue* q)
{
	if ( q->Rear > q->Front ) return (q->Size - q->Rear + q->Front - 1);
	else if ( q->Rear < q->Front ) return (q->Front - q->Rear - 1);
	else return (q->Size - 1);
}


void CqSync(TCircularQueue* q, uint front, uint rear)
{
	q->Front = front;
	q->Rear = rear;
}


bool CqPutByte(TCircularQueue* q, char data)
{
	if ( (q->Rear+1) % q->Size == q->Front ) return FALSE;

	q->Rear = (q->Rear+1) % q->Size;
	q->Buf[q->Rear] = data;
	return TRUE;
}


bool CqGetByte(TCircularQueue* q, char* data)
{
	if (q->Front == q->Rear) return FALSE;

	q->Front = (q->Front+1) % q->Size;
	*data = q->Buf[q->Front];
	return TRUE;
}


void CqPutBuf(TCircularQueue* q, void* buf, uint bufsize, uint* realsize)
{
	uint i, size;

	if ( CqUnusedSize(q) < bufsize )
		size = CqUnusedSize(q);
	else size = bufsize;

	if ( realsize )
		*realsize = size;

	for ( i=0 ; i<size ; i++ )
	{
    	q->Rear++;
    	if ( q->Rear >= q->Size ) q->Rear %= q->Size;
    	q->Buf[q->Rear] = ((char*)buf)[i];
	}
}


void CqGetBuf(TCircularQueue* q, void* buf, uint bufsize, uint* realsize)
{
	uint i, size;

	if ( CqUsedSize(q) < bufsize )
		size = CqUsedSize(q);
	else size = bufsize;

	if ( realsize )
		*realsize = size;

	for ( i=0 ; i<size ; i++ )
	{
    	q->Front++;
    	if ( q->Front >= q->Size ) q->Front %= q->Size;
    	((char*)buf)[i] = q->Buf[q->Front];
    }
}


void CqMoveBuf(TCircularQueue* q1, TCircularQueue* q2, uint size, uint* realsize)
{
	char temp[255];

	if ( size > CqUsedSize(q1) ) size = CqUsedSize(q1);
	if ( size > CqUnusedSize(q2) ) size = CqUnusedSize(q2);

	if ( realsize ) *realsize = size;

	while ( size > 0 )
	{
		if ( size < 255 )
		{
			CqGetBuf(q1, temp, size, NULL);
			CqPutBuf(q2, temp, size, NULL);

			size = 0;
		}
		else
		{
			CqGetBuf(q1, temp, 255, NULL);
			CqPutBuf(q2, temp, 255, NULL);

			size -= 255;
		}
	}
}


bool CqPutBufAll(TCircularQueue* q, void* buf, uint bufsize)
{
	if ( CqUnusedSize(q) >= bufsize )
	{
		CqPutBuf(q, buf, bufsize, NULL);
		return TRUE;
	}
	else return FALSE;
}


bool CqGetBufAll(TCircularQueue* q, void* buf, uint bufsize)
{
	if ( CqUsedSize(q) <= bufsize )
	{
		CqGetBuf(q, buf, bufsize, NULL);
		return TRUE;
	}
	else return FALSE;
}


bool CqMoveBufAll(TCircularQueue* q1, TCircularQueue* q2)
{
	if ( CqUsedSize(q1) <= CqUnusedSize(q2) )
	{
		CqMoveBuf(q1, q2, 0xFFFFFFFF, NULL);
		return TRUE;
	}
	else return FALSE;
}


char* CqParse(TCircularQueue* q, char delimitator)
{
	char ch;
	static uint len = 0;
	static char buf[255];

	while ( CqGetByte(q, &ch) )
	{
		if ( ch != delimitator )
		{
			buf[len++] = ch;
		}
		else
		{
			buf[len] = '\0';
			len = 0;
			return buf;
		}
	}

	return NULL;
}


// must num > 1
char* CqParse2(TCircularQueue* q, char* delimitators, int num)
{
	char ch;
	static uint match_count = 0, pos = 0, len = 0;
	static char buf[255];

	while ( CqGetByte(q, &ch) )
	{
	    if ( ch == delimitators[match_count] )
	    {
	        match_count++;

	        if ( match_count == 1 )
	        {
	            pos = len;
	        }
            else if ( match_count == num )
        	{
        	    buf[pos] = '\0';
        	    len = 0;
        	    match_count = 0;
        	    return buf;
        	}
	    }
	    else match_count = 0;

		buf[len++] = ch;
	}

	return NULL;
}
